package com.zjx.cloud.alibaba.sentinel;
import com.alibaba.csp.sentinel.adapter.gateway.common.SentinelGatewayConstants;
import com.alibaba.csp.sentinel.adapter.gateway.common.api.ApiDefinition;
import com.alibaba.csp.sentinel.adapter.gateway.common.api.ApiPathPredicateItem;
import com.alibaba.csp.sentinel.adapter.gateway.common.api.ApiPredicateItem;
import com.alibaba.csp.sentinel.adapter.gateway.common.api.GatewayApiDefinitionManager;
import com.alibaba.csp.sentinel.adapter.gateway.common.rule.GatewayFlowRule;
import com.alibaba.csp.sentinel.adapter.gateway.common.rule.GatewayRuleManager;
import com.alibaba.csp.sentinel.adapter.gateway.sc.SentinelGatewayFilter;
import com.alibaba.csp.sentinel.adapter.gateway.sc.callback.GatewayCallbackManager;
import com.alibaba.csp.sentinel.adapter.gateway.sc.exception.SentinelGatewayBlockExceptionHandler;
import org.springframework.beans.factory.ObjectProvider;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.core.annotation.Order;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.codec.ServerCodecConfigurer;
import org.springframework.web.reactive.function.server.ServerResponse;
import org.springframework.web.reactive.result.view.ViewResolver;

import javax.annotation.PostConstruct;
import java.util.*;

@Configuration
public class SentinelGatewayConfiguration {
    private final List<ViewResolver> viewResolvers;    // 视图解析器
    private final ServerCodecConfigurer serverCodecConfigurer;  // 编码器

    public SentinelGatewayConfiguration(ObjectProvider<List<ViewResolver>> viewResolversProvider,
                                        ServerCodecConfigurer serverCodecConfigurer) {
        this.viewResolvers = viewResolversProvider.getIfAvailable(Collections::emptyList);
        this.serverCodecConfigurer = serverCodecConfigurer;
    }

    @Bean
    @Order(Ordered.HIGHEST_PRECEDENCE)
    public SentinelGatewayBlockExceptionHandler sentinelGatewayBlockExceptionHandler() {
        // 注册服务降级（异常）处理器
        return new SentinelGatewayBlockExceptionHandler(viewResolvers, serverCodecConfigurer);
    }

    @Bean
    @Order(-1)
    public GlobalFilter sentinelGatewayFilter() {
        return new SentinelGatewayFilter();
    }

    @PostConstruct
    public void doInit() {
        initCustomizedApis();  // 自定义 api 维度
        initGatewayRules();    // 路由规则 维度
    }

// 自定义API分组
private void initCustomizedApis() {
    Set<ApiDefinition> definitions = new HashSet<>();
    ApiDefinition api01 = new ApiDefinition("my-group-01")
            .setPredicateItems(new HashSet<ApiPredicateItem>() {{
                // url前缀匹配
                add(new ApiPathPredicateItem().setPattern("/my-app-2/test/echo2/**")
                        // 匹配策略【前缀、正则、精确（默认）】
                        .setMatchStrategy(SentinelGatewayConstants.URL_MATCH_STRATEGY_PREFIX));
                // 精确指定url
                add(new ApiPathPredicateItem().setPattern("/my-app-2/test/echo3/hi"));
            }});
    definitions.add(api01);
    GatewayApiDefinitionManager.loadApiDefinitions(definitions);
}
private void initGatewayRules() {
    Set<GatewayFlowRule> rules = new HashSet<>();
    // 限制每秒1次请求 -- 根据 Gateway 配置文件的 route 名称
    rules.add(new GatewayFlowRule("cloud-alibaba-provider")
            .setCount(1)
            .setIntervalSec(1)
    );
    // 限制每秒3次请求 -- 根据自定义API分组
    rules.add(new GatewayFlowRule("my-group-01")
            // 自定义，需要指定模式
            .setResourceMode(SentinelGatewayConstants.RESOURCE_MODE_CUSTOM_API_NAME)
            .setCount(3)
            .setIntervalSec(1)
    );
    GatewayRuleManager.loadRules(rules); // 注册规则

    // 触发限流，设置处理器处理
    GatewayCallbackManager.setBlockHandler((exchange, throwable) -> {
        Map<String, Object> map = new HashMap<>();
        map.put("errCode", throwable.getMessage());
        map.put("errMsg", "请求太过频繁！");
        map.put("请求地址",exchange.getRequest().getURI());
        return ServerResponse.status(HttpStatus.TOO_MANY_REQUESTS)
                .contentType(MediaType.APPLICATION_JSON)
                .bodyValue(map);
    });
}

}