package com.zjx.cloud.alibaba.config;

import lombok.extern.slf4j.Slf4j;
import org.springframework.cloud.gateway.filter.GatewayFilterChain;
import org.springframework.cloud.gateway.filter.GlobalFilter;
import org.springframework.core.Ordered;
import org.springframework.http.HttpStatus;
import org.springframework.http.server.reactive.ServerHttpRequest;
import org.springframework.http.server.reactive.ServerHttpResponse;
import org.springframework.stereotype.Component;
import org.springframework.web.server.ServerWebExchange;
import reactor.core.publisher.Mono;

import java.net.URI;
import java.util.Optional;

/**
 * 自定义全局过滤器，计算接口处理时间
 */
@Component
@Slf4j
public class MyCustomGlobalFilter implements GlobalFilter, Ordered {
    public static final String BEGIN_TIME = "beginTime";
    // 自定义过滤器逻辑
    @Override
    public Mono<Void> filter(ServerWebExchange exchange, GatewayFilterChain chain) {
        // ServerWebExchange 存放着请求和响应信息，如下示例
        // ServerHttpRequest request = exchange.getRequest();
        // ServerHttpResponse response = exchange.getResponse();
        // 设置开始处理请求时间
        exchange.getAttributes().put(BEGIN_TIME, System.currentTimeMillis());
        // 放行，执行下一步,开启统计线程
        return chain.filter(exchange).then(Mono.fromRunnable(() -> {
            // url 不存在，给前端提示一下
            if (exchange.getResponse().getStatusCode() == HttpStatus.NOT_FOUND)
                throw new RuntimeException("您访问的URL不存在！");
            // 这里面写请求处理后的逻辑
            URI uri = exchange.getRequest().getURI();
            Optional.ofNullable((Long) exchange.getAttribute(BEGIN_TIME))
                    .ifPresent(startTime -> {
                                // 计算耗时
                                log.info("访问主机：{} 端口:{} url：{} 参数:{}", uri.getHost(),
                                        uri.getPort(), uri.getPath(), uri.getQuery());
                                log.info("访问时长：{}毫秒", System.currentTimeMillis() - startTime);
                            });}));
    }
    // 过滤器排序，越小优先级越高
    @Override
    public int getOrder() { return -1; }
}